/*
 * Decompiled with CFR 0.152.
 */
package uk.org.okapibarcode.backend;

import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.util.Arrays;
import uk.org.okapibarcode.backend.HumanReadableLocation;
import uk.org.okapibarcode.backend.OkapiInputException;
import uk.org.okapibarcode.backend.OkapiInternalException;
import uk.org.okapibarcode.backend.ReedSolomon;
import uk.org.okapibarcode.backend.Symbol;
import uk.org.okapibarcode.util.Strings;

public class QrCode
extends Symbol {
    private static final char[] RHODIUM = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', ' ', '$', '%', '*', '+', '-', '.', '/', ':'};
    private static final int[] QR_DATA_CODEWORDS_L = new int[]{19, 34, 55, 80, 108, 136, 156, 194, 232, 274, 324, 370, 428, 461, 523, 589, 647, 721, 795, 861, 932, 1006, 1094, 1174, 1276, 1370, 1468, 1531, 1631, 1735, 1843, 1955, 2071, 2191, 2306, 2434, 2566, 2702, 2812, 2956};
    private static final int[] QR_DATA_CODEWORDS_M = new int[]{16, 28, 44, 64, 86, 108, 124, 154, 182, 216, 254, 290, 334, 365, 415, 453, 507, 563, 627, 669, 714, 782, 860, 914, 1000, 1062, 1128, 1193, 1267, 1373, 1455, 1541, 1631, 1725, 1812, 1914, 1992, 2102, 2216, 2334};
    private static final int[] QR_DATA_CODEWORDS_Q = new int[]{13, 22, 34, 48, 62, 76, 88, 110, 132, 154, 180, 206, 244, 261, 295, 325, 367, 397, 445, 485, 512, 568, 614, 664, 718, 754, 808, 871, 911, 985, 1033, 1115, 1171, 1231, 1286, 1354, 1426, 1502, 1582, 1666};
    private static final int[] QR_DATA_CODEWORDS_H = new int[]{9, 16, 26, 36, 46, 60, 66, 86, 100, 122, 140, 158, 180, 197, 223, 253, 283, 313, 341, 385, 406, 442, 464, 514, 538, 596, 628, 661, 701, 745, 793, 845, 901, 961, 986, 1054, 1096, 1142, 1222, 1276};
    private static final int[] QR_BLOCKS_L = new int[]{1, 1, 1, 1, 1, 2, 2, 2, 2, 4, 4, 4, 4, 4, 6, 6, 6, 6, 7, 8, 8, 9, 9, 10, 12, 12, 12, 13, 14, 15, 16, 17, 18, 19, 19, 20, 21, 22, 24, 25};
    private static final int[] QR_BLOCKS_M = new int[]{1, 1, 1, 2, 2, 4, 4, 4, 5, 5, 5, 8, 9, 9, 10, 10, 11, 13, 14, 16, 17, 17, 18, 20, 21, 23, 25, 26, 28, 29, 31, 33, 35, 37, 38, 40, 43, 45, 47, 49};
    private static final int[] QR_BLOCKS_Q = new int[]{1, 1, 2, 2, 4, 4, 6, 6, 8, 8, 8, 10, 12, 16, 12, 17, 16, 18, 21, 20, 23, 23, 25, 27, 29, 34, 34, 35, 38, 40, 43, 45, 48, 51, 53, 56, 59, 62, 65, 68};
    private static final int[] QR_BLOCKS_H = new int[]{1, 1, 2, 4, 4, 4, 5, 6, 8, 8, 11, 11, 16, 16, 18, 16, 19, 21, 25, 25, 25, 34, 30, 32, 35, 37, 40, 42, 45, 48, 51, 54, 57, 60, 63, 66, 70, 74, 77, 81};
    private static final int[] QR_TOTAL_CODEWORDS = new int[]{26, 44, 70, 100, 134, 172, 196, 242, 292, 346, 404, 466, 532, 581, 655, 733, 815, 901, 991, 1085, 1156, 1258, 1364, 1474, 1588, 1706, 1828, 1921, 2051, 2185, 2323, 2465, 2611, 2761, 2876, 3034, 3196, 3362, 3532, 3706};
    private static final int[] QR_SIZES = new int[]{21, 25, 29, 33, 37, 41, 45, 49, 53, 57, 61, 65, 69, 73, 77, 81, 85, 89, 93, 97, 101, 105, 109, 113, 117, 121, 125, 129, 133, 137, 141, 145, 149, 153, 157, 161, 165, 169, 173, 177};
    private static final int[] QR_ALIGN_LOOPSIZE = new int[]{0, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7};
    private static final int[] QR_TABLE_E1 = new int[]{6, 18, 0, 0, 0, 0, 0, 6, 22, 0, 0, 0, 0, 0, 6, 26, 0, 0, 0, 0, 0, 6, 30, 0, 0, 0, 0, 0, 6, 34, 0, 0, 0, 0, 0, 6, 22, 38, 0, 0, 0, 0, 6, 24, 42, 0, 0, 0, 0, 6, 26, 46, 0, 0, 0, 0, 6, 28, 50, 0, 0, 0, 0, 6, 30, 54, 0, 0, 0, 0, 6, 32, 58, 0, 0, 0, 0, 6, 34, 62, 0, 0, 0, 0, 6, 26, 46, 66, 0, 0, 0, 6, 26, 48, 70, 0, 0, 0, 6, 26, 50, 74, 0, 0, 0, 6, 30, 54, 78, 0, 0, 0, 6, 30, 56, 82, 0, 0, 0, 6, 30, 58, 86, 0, 0, 0, 6, 34, 62, 90, 0, 0, 0, 6, 28, 50, 72, 94, 0, 0, 6, 26, 50, 74, 98, 0, 0, 6, 30, 54, 78, 102, 0, 0, 6, 28, 54, 80, 106, 0, 0, 6, 32, 58, 84, 110, 0, 0, 6, 30, 58, 86, 114, 0, 0, 6, 34, 62, 90, 118, 0, 0, 6, 26, 50, 74, 98, 122, 0, 6, 30, 54, 78, 102, 126, 0, 6, 26, 52, 78, 104, 130, 0, 6, 30, 56, 82, 108, 134, 0, 6, 34, 60, 86, 112, 138, 0, 6, 30, 58, 86, 114, 142, 0, 6, 34, 62, 90, 118, 146, 0, 6, 30, 54, 78, 102, 126, 150, 6, 24, 50, 76, 102, 128, 154, 6, 28, 54, 80, 106, 132, 158, 6, 32, 58, 84, 110, 136, 162, 6, 26, 54, 82, 110, 138, 166, 6, 30, 58, 86, 114, 142, 170};
    private static final int[] QR_ANNEX_C = new int[]{21522, 20773, 24188, 23371, 17913, 16590, 20375, 19104, 30660, 29427, 32170, 30877, 26159, 25368, 27713, 26998, 5769, 5054, 7399, 6608, 1890, 597, 3340, 2107, 13663, 12392, 16177, 14854, 9396, 8579, 11994, 11245};
    private static final int[] QR_ANNEX_D = new int[]{31892, 34236, 39577, 42195, 48118, 51042, 55367, 58893, 63784, 68472, 70749, 76311, 79154, 84390, 87683, 92361, 96236, 102084, 102881, 110507, 110734, 117786, 119615, 126325, 127568, 133589, 136944, 141498, 145311, 150283, 152622, 158308, 161089, 167017};
    protected int minVersion = 1;
    protected int preferredVersion;
    protected EccLevel preferredEccLevel = EccLevel.L;
    protected boolean improveEccLevelIfPossible = true;
    protected boolean forceByteCompaction;

    public QrCode() {
        this.humanReadableLocation = HumanReadableLocation.NONE;
    }

    public void setPreferredVersion(int version) {
        if (version < 1 || version > 40) {
            throw new IllegalArgumentException("Invalid QR Code version: " + version);
        }
        this.preferredVersion = version;
    }

    public int getPreferredVersion() {
        return this.preferredVersion;
    }

    public void setPreferredEccLevel(EccLevel preferredEccLevel) {
        this.preferredEccLevel = preferredEccLevel;
    }

    public EccLevel getPreferredEccLevel() {
        return this.preferredEccLevel;
    }

    public void setForceByteCompaction(boolean forceByteCompaction) {
        this.forceByteCompaction = forceByteCompaction;
    }

    public boolean getForceByteCompaction() {
        return this.forceByteCompaction;
    }

    @Override
    public boolean supportsGs1() {
        return true;
    }

    @Override
    public boolean supportsEci() {
        return true;
    }

    @Override
    protected void encode() {
        int max_cw;
        boolean gs1;
        int i;
        this.eciProcess();
        if (this.eciMode == 20) {
            Charset sjis = Charset.forName("Shift_JIS");
            this.inputData = new int[this.content.length()];
            for (i = 0; i < this.inputData.length; ++i) {
                int value;
                CharBuffer buffer = CharBuffer.wrap(this.content, i, i + 1);
                byte[] bytes = sjis.encode(buffer).array();
                this.inputData[i] = value = bytes.length == 2 && bytes[1] != 0 ? (bytes[0] & 0xFF) << 8 | bytes[1] & 0xFF : bytes[0];
            }
        }
        if (this.inputDataType == Symbol.DataType.GS1) {
            gs1 = true;
        } else if (this.inputDataType == Symbol.DataType.ECI && this.inputData.length > 0 && this.inputData[0] == -1) {
            gs1 = true;
            this.inputData = Arrays.copyOfRange(this.inputData, 1, this.inputData.length);
        } else {
            gs1 = false;
        }
        QrMode[] inputMode = new QrMode[this.inputData.length];
        QrCode.defineMode(inputMode, this.inputData, this.forceByteCompaction);
        int est_binlen = QrCode.getBinaryLength(40, inputMode, this.inputData, gs1, this.eciMode);
        EccLevel ecc_level = this.preferredEccLevel;
        switch (ecc_level.ordinal()) {
            default: {
                max_cw = 2956;
                break;
            }
            case 1: {
                max_cw = 2334;
                break;
            }
            case 2: {
                max_cw = 1666;
                break;
            }
            case 3: {
                max_cw = 1276;
            }
        }
        if (est_binlen > 8 * max_cw) {
            throw new OkapiInputException("Input too long for selected error correction level");
        }
        int version = 40;
        for (int candidate = 40; candidate >= this.minVersion; --candidate) {
            int[] dataCodewords;
            switch (ecc_level.ordinal()) {
                default: {
                    dataCodewords = QR_DATA_CODEWORDS_L;
                    break;
                }
                case 1: {
                    dataCodewords = QR_DATA_CODEWORDS_M;
                    break;
                }
                case 2: {
                    dataCodewords = QR_DATA_CODEWORDS_Q;
                    break;
                }
                case 3: {
                    dataCodewords = QR_DATA_CODEWORDS_H;
                }
            }
            int proposedBinLen = QrCode.getBinaryLength(candidate, inputMode, this.inputData, gs1, this.eciMode);
            if (8 * dataCodewords[candidate - 1] < proposedBinLen) continue;
            version = candidate;
            est_binlen = proposedBinLen;
        }
        inputMode = QrCode.applyOptimisation(version, inputMode);
        if (this.preferredVersion > 0) {
            if (this.preferredVersion > version) {
                version = this.preferredVersion;
                est_binlen = QrCode.getBinaryLength(this.preferredVersion, inputMode, this.inputData, gs1, this.eciMode);
                inputMode = QrCode.applyOptimisation(version, inputMode);
            }
            if (this.preferredVersion < version) {
                throw new OkapiInputException("Input too long for selected symbol size");
            }
        }
        if (this.improveEccLevelIfPossible) {
            if (est_binlen <= QR_DATA_CODEWORDS_M[version - 1] * 8) {
                ecc_level = EccLevel.M;
            }
            if (est_binlen <= QR_DATA_CODEWORDS_Q[version - 1] * 8) {
                ecc_level = EccLevel.Q;
            }
            if (est_binlen <= QR_DATA_CODEWORDS_H[version - 1] * 8) {
                ecc_level = EccLevel.H;
            }
        }
        int targetCwCount = QR_DATA_CODEWORDS_L[version - 1];
        int blocks = QR_BLOCKS_L[version - 1];
        switch (ecc_level.ordinal()) {
            case 1: {
                targetCwCount = QR_DATA_CODEWORDS_M[version - 1];
                blocks = QR_BLOCKS_M[version - 1];
                break;
            }
            case 2: {
                targetCwCount = QR_DATA_CODEWORDS_Q[version - 1];
                blocks = QR_BLOCKS_Q[version - 1];
                break;
            }
            case 3: {
                targetCwCount = QR_DATA_CODEWORDS_H[version - 1];
                blocks = QR_BLOCKS_H[version - 1];
            }
        }
        int[] datastream = new int[targetCwCount + 1];
        int[] fullstream = new int[QR_TOTAL_CODEWORDS[version - 1] + 1];
        this.qrBinary(datastream, version, targetCwCount, inputMode, this.inputData, gs1, this.eciMode, est_binlen);
        QrCode.addEcc(fullstream, datastream, version, targetCwCount, blocks);
        this.infoLine("Version: " + version);
        this.infoLine("ECC Level: " + ecc_level.name());
        int size = QR_SIZES[version - 1];
        int[] grid = new int[size * size];
        QrCode.setupGrid(grid, size, version);
        QrCode.populateGrid(grid, size, fullstream, QR_TOTAL_CODEWORDS[version - 1]);
        if (version >= 7) {
            QrCode.addVersionInfo(grid, size, version);
        }
        int bitmask = QrCode.applyBitmask(grid, size, ecc_level, this.encodeInfo);
        this.infoLine("Mask Pattern: " + QrCode.maskToString(bitmask));
        QrCode.addFormatInfo(grid, size, ecc_level, bitmask);
        this.customize(grid, size);
        this.readable = "";
        this.pattern = new String[size];
        this.row_count = size;
        this.row_height = new int[size];
        for (i = 0; i < size; ++i) {
            StringBuilder bin = new StringBuilder(size);
            for (int j = 0; j < size; ++j) {
                if ((grid[i * size + j] & 1) != 0) {
                    bin.append('1');
                    continue;
                }
                bin.append('0');
            }
            this.pattern[i] = QrCode.bin2pat(bin);
            this.row_height[i] = this.moduleWidth;
        }
    }

    private static void defineMode(QrMode[] inputMode, int[] inputData, boolean forceByteCompaction) {
        if (forceByteCompaction) {
            Arrays.fill((Object[])inputMode, 0, inputData.length, (Object)QrMode.BINARY);
            return;
        }
        for (int i = 0; i < inputData.length; ++i) {
            if (inputData[i] > 255) {
                inputMode[i] = QrMode.KANJI;
                continue;
            }
            inputMode[i] = QrMode.BINARY;
            if (QrCode.isAlpha(inputData[i])) {
                inputMode[i] = QrMode.ALPHANUM;
            }
            if (inputData[i] == -1) {
                inputMode[i] = QrMode.ALPHANUM;
            }
            if (!QrCode.isNumeric(inputData[i])) continue;
            inputMode[i] = QrMode.NUMERIC;
        }
    }

    private static int getBinaryLength(int version, QrMode[] inputModeUnoptimized, int[] inputData, boolean gs1, int eciMode) {
        int inputLength = inputModeUnoptimized.length;
        int count = 0;
        int percent = 0;
        QrMode[] inputMode = QrCode.applyOptimisation(version, inputModeUnoptimized);
        QrMode currentMode = QrMode.NULL;
        if (gs1) {
            count += 4;
        }
        if (eciMode != 3) {
            count += 12;
        }
        for (int i = 0; i < inputLength; ++i) {
            if (inputMode[i] == currentMode) continue;
            count += 4;
            block0 : switch (inputMode[i].ordinal()) {
                case 1: {
                    count += QrCode.tribus(version, 8, 10, 12);
                    count += QrCode.blockLength(i, inputMode) * 13;
                    break;
                }
                case 2: {
                    int j;
                    count += QrCode.tribus(version, 8, 16, 16);
                    int max = i + QrCode.blockLength(i, inputMode);
                    for (j = i; j < max; ++j) {
                        if (inputData[j] > 255) {
                            count += 16;
                            continue;
                        }
                        count += 8;
                    }
                    break;
                }
                case 3: {
                    int j;
                    count += QrCode.tribus(version, 9, 11, 13);
                    int alphaLength = QrCode.blockLength(i, inputMode);
                    if (gs1) {
                        for (j = i; j < i + alphaLength; ++j) {
                            if (inputData[j] != 37) continue;
                            ++percent;
                        }
                    }
                    switch ((alphaLength += percent) % 2) {
                        case 0: {
                            count += alphaLength / 2 * 11;
                            break;
                        }
                        case 1: {
                            count += (alphaLength - 1) / 2 * 11;
                            count += 6;
                        }
                    }
                    break;
                }
                case 4: {
                    count += QrCode.tribus(version, 10, 12, 14);
                    int length = QrCode.blockLength(i, inputMode);
                    switch (length % 3) {
                        case 0: {
                            count += length / 3 * 10;
                            break block0;
                        }
                        case 1: {
                            count += (length - 1) / 3 * 10;
                            count += 4;
                            break block0;
                        }
                        case 2: {
                            count += (length - 2) / 3 * 10;
                            count += 7;
                        }
                    }
                }
            }
            currentMode = inputMode[i];
        }
        return count;
    }

    private static QrMode[] applyOptimisation(int version, QrMode[] inputMode) {
        int i;
        int inputLength = inputMode.length;
        int blockCount = 0;
        QrMode currentMode = QrMode.NULL;
        for (i = 0; i < inputLength; ++i) {
            if (inputMode[i] == currentMode) continue;
            currentMode = inputMode[i];
            ++blockCount;
        }
        int[] blockLength = new int[blockCount];
        QrMode[] blockMode = new QrMode[blockCount];
        int j = -1;
        currentMode = QrMode.NULL;
        for (i = 0; i < inputLength; ++i) {
            if (inputMode[i] != currentMode) {
                blockLength[++j] = 1;
                blockMode[j] = inputMode[i];
                currentMode = inputMode[i];
                continue;
            }
            int n = j;
            blockLength[n] = blockLength[n] + 1;
        }
        if (blockCount > 1) {
            int min;
            boolean returns;
            for (i = 0; i < blockCount - 1; ++i) {
                boolean bl = returns = i + 2 < blockMode.length && blockMode[i + 2] == blockMode[i];
                if (blockMode[i] == QrMode.BINARY) {
                    switch (blockMode[i + 1].ordinal()) {
                        case 1: {
                            int n = min = returns ? QrCode.tribus(version, 12, 18, 20) : QrCode.tribus(version, 6, 8, 8);
                            if (blockLength[i + 1] * 2 >= min) break;
                            blockMode[i + 1] = QrMode.BINARY;
                            break;
                        }
                        case 3: {
                            int n = min = returns ? QrCode.tribus(version, 7, 11, 12) : QrCode.tribus(version, 4, 5, 6);
                            if (blockLength[i + 1] >= min) break;
                            blockMode[i + 1] = QrMode.BINARY;
                            break;
                        }
                        case 4: {
                            int n = min = returns ? QrCode.tribus(version, 4, 7, 7) : QrCode.tribus(version, 3, 3, 4);
                            if (blockLength[i + 1] >= min) break;
                            blockMode[i + 1] = QrMode.BINARY;
                        }
                    }
                }
                if (blockMode[i] != QrMode.ALPHANUM || blockMode[i + 1] != QrMode.NUMERIC) continue;
                int n = min = returns ? QrCode.tribus(version, 9, 12, 15) : QrCode.tribus(version, 6, 9, 9);
                if (blockLength[i + 1] >= min) continue;
                blockMode[i + 1] = QrMode.ALPHANUM;
            }
            for (i = blockCount - 1; i > 0; --i) {
                boolean bl = returns = i - 2 >= 0 && blockMode[i - 2] == blockMode[i];
                if (blockMode[i] == QrMode.BINARY) {
                    switch (blockMode[i - 1].ordinal()) {
                        case 1: {
                            int n = min = returns ? QrCode.tribus(version, 12, 18, 20) : QrCode.tribus(version, 6, 8, 8);
                            if (blockLength[i - 1] * 2 >= min) break;
                            blockMode[i - 1] = QrMode.BINARY;
                            break;
                        }
                        case 3: {
                            int n = min = returns ? QrCode.tribus(version, 7, 11, 12) : QrCode.tribus(version, 4, 5, 6);
                            if (blockLength[i - 1] >= min) break;
                            blockMode[i - 1] = QrMode.BINARY;
                            break;
                        }
                        case 4: {
                            int n = min = returns ? QrCode.tribus(version, 4, 7, 7) : QrCode.tribus(version, 3, 3, 4);
                            if (blockLength[i - 1] >= min) break;
                            blockMode[i - 1] = QrMode.BINARY;
                        }
                    }
                }
                if (blockMode[i] != QrMode.ALPHANUM || blockMode[i - 1] != QrMode.NUMERIC) continue;
                int n = min = returns ? QrCode.tribus(version, 9, 12, 15) : QrCode.tribus(version, 6, 9, 9);
                if (blockLength[i - 1] >= min) continue;
                blockMode[i - 1] = QrMode.ALPHANUM;
            }
        }
        QrMode[] optimized = new QrMode[inputMode.length];
        j = 0;
        for (int block = 0; block < blockCount; ++block) {
            currentMode = blockMode[block];
            for (i = 0; i < blockLength[block]; ++i) {
                optimized[j] = currentMode;
                ++j;
            }
        }
        return optimized;
    }

    private static int blockLength(int start, QrMode[] inputMode) {
        int i;
        QrMode mode = inputMode[start];
        for (i = start + 1; i < inputMode.length && inputMode[i] == mode; ++i) {
        }
        return i - start;
    }

    private static int tribus(int version, int a, int b, int c) {
        if (version < 10) {
            return a;
        }
        if (version < 27) {
            return b;
        }
        return c;
    }

    private static boolean isAlpha(int c) {
        return c >= 48 && c <= 57 || c >= 65 && c <= 90 || c == 32 || c == 36 || c == 37 || c == 42 || c == 43 || c == 45 || c == 46 || c == 47 || c == 58;
    }

    private static boolean isNumeric(int c) {
        return c >= 48 && c <= 57;
    }

    private static String maskToString(int mask) {
        switch (mask) {
            case 0: {
                return "000";
            }
            case 1: {
                return "001";
            }
            case 2: {
                return "010";
            }
            case 3: {
                return "011";
            }
            case 4: {
                return "100";
            }
            case 5: {
                return "101";
            }
            case 6: {
                return "110";
            }
            case 7: {
                return "111";
            }
        }
        return "000";
    }

    private void qrBinary(int[] datastream, int version, int target_binlen, QrMode[] inputMode, int[] inputData, boolean gs1, int eciMode, int est_binlen) {
        int i;
        int position = 0;
        int reserved = est_binlen + 12;
        StringBuilder binary = new StringBuilder(reserved);
        if (gs1) {
            binary.append("0101");
        }
        if (eciMode != 3) {
            binary.append("0111");
            if (eciMode <= 127) {
                Strings.binaryAppend(binary, eciMode, 8);
            } else if (eciMode <= 16383) {
                Strings.binaryAppend(binary, 32768 + eciMode, 16);
            } else {
                Strings.binaryAppend(binary, 0xC00000 + eciMode, 24);
            }
        }
        this.info("Encoding: ");
        while (position < inputMode.length) {
            QrMode data_block = inputMode[position];
            int short_data_block_length = 0;
            while (++short_data_block_length + position < inputMode.length && inputMode[position + short_data_block_length] == data_block) {
            }
            switch (data_block.ordinal()) {
                case 1: {
                    binary.append("1000");
                    Strings.binaryAppend(binary, short_data_block_length, QrCode.tribus(version, 8, 10, 12));
                    this.info("KNJI ");
                    for (i = 0; i < short_data_block_length; ++i) {
                        int jis = inputData[position + i];
                        if (jis >= 33088 && jis <= 40956) {
                            jis -= 33088;
                        } else if (jis >= 57408 && jis <= 60351) {
                            jis -= 49472;
                        }
                        int prod = (jis >> 8) * 192 + (jis & 0xFF);
                        Strings.binaryAppend(binary, prod, 13);
                        this.infoSpace(prod);
                    }
                    break;
                }
                case 2: {
                    int b;
                    binary.append("0100");
                    int bytes = 0;
                    for (i = 0; i < short_data_block_length; ++i) {
                        b = inputData[position + i];
                        bytes += b > 255 ? 2 : 1;
                    }
                    Strings.binaryAppend(binary, bytes, QrCode.tribus(version, 8, 16, 16));
                    this.info("BYTE ");
                    for (i = 0; i < short_data_block_length; ++i) {
                        b = inputData[position + i];
                        if (b > 255) {
                            int b1 = b >> 8;
                            int b2 = b & 0xFF;
                            Strings.binaryAppend(binary, b1, 8);
                            this.infoSpace(b1);
                            Strings.binaryAppend(binary, b2, 8);
                            this.infoSpace(b2);
                            continue;
                        }
                        if (b == -1) {
                            b = 29;
                        }
                        Strings.binaryAppend(binary, b, 8);
                        this.infoSpace(b);
                    }
                    break;
                }
                case 3: {
                    int prod;
                    int first;
                    binary.append("0010");
                    int percentCount = 0;
                    if (gs1) {
                        for (i = 0; i < short_data_block_length; ++i) {
                            if (inputData[position + i] != 37) continue;
                            ++percentCount;
                        }
                    }
                    int[] inputExpanded = new int[short_data_block_length + percentCount];
                    percentCount = 0;
                    for (i = 0; i < short_data_block_length; ++i) {
                        int c = inputData[position + i];
                        if (c == -1) {
                            inputExpanded[i + percentCount] = 37;
                            continue;
                        }
                        inputExpanded[i + percentCount] = c;
                        if (!gs1 || c != 37) continue;
                        inputExpanded[i + ++percentCount] = c;
                    }
                    Strings.binaryAppend(binary, inputExpanded.length, QrCode.tribus(version, 9, 11, 13));
                    this.info("ALPH ");
                    i = 0;
                    while (i + 1 < inputExpanded.length) {
                        first = uk.org.okapibarcode.util.Arrays.positionOf((char)inputExpanded[i], RHODIUM);
                        int second = uk.org.okapibarcode.util.Arrays.positionOf((char)inputExpanded[i + 1], RHODIUM);
                        prod = first * 45 + second;
                        int count = 2;
                        Strings.binaryAppend(binary, prod, 1 + 5 * count);
                        this.infoSpace(prod);
                        i += 2;
                    }
                    if (inputExpanded.length % 2 == 0) break;
                    int prod2 = first = uk.org.okapibarcode.util.Arrays.positionOf((char)inputExpanded[inputExpanded.length - 1], RHODIUM);
                    int count = 1;
                    Strings.binaryAppend(binary, prod2, 1 + 5 * count);
                    this.infoSpace(prod2);
                    break;
                }
                case 4: {
                    int count;
                    int prod;
                    int first;
                    binary.append("0001");
                    Strings.binaryAppend(binary, short_data_block_length, QrCode.tribus(version, 10, 12, 14));
                    this.info("NUMB ");
                    for (i = 0; i < short_data_block_length; i += count) {
                        first = Character.getNumericValue(inputData[position + i]);
                        count = 1;
                        prod = first;
                        if (i + 1 < short_data_block_length) {
                            int second = Character.getNumericValue(inputData[position + i + 1]);
                            count = 2;
                            prod = prod * 10 + second;
                            if (i + 2 < short_data_block_length) {
                                int third = Character.getNumericValue(inputData[position + i + 2]);
                                count = 3;
                                prod = prod * 10 + third;
                            }
                        }
                        Strings.binaryAppend(binary, prod, 1 + 3 * count);
                        this.infoSpace(prod);
                    }
                    break;
                }
            }
            position += short_data_block_length;
        }
        this.infoLine();
        binary.append("0000");
        int current_binlen = binary.length();
        int padbits = 8 - current_binlen % 8;
        if (padbits == 8) {
            padbits = 0;
        }
        int current_bytes = (current_binlen + padbits) / 8;
        for (i = 0; i < padbits; ++i) {
            binary.append('0');
        }
        for (i = 0; i < current_bytes; ++i) {
            datastream[i] = 0;
            for (int p = 0; p < 8; ++p) {
                if (binary.charAt(i * 8 + p) != '1') continue;
                int n = i;
                datastream[n] = datastream[n] + (128 >> p);
            }
        }
        boolean toggle = false;
        for (i = current_bytes; i < target_binlen; ++i) {
            if (!toggle) {
                datastream[i] = 236;
                toggle = true;
                continue;
            }
            datastream[i] = 17;
            toggle = false;
        }
        this.info("Codewords: ");
        for (i = 0; i < target_binlen; ++i) {
            this.infoSpace(datastream[i]);
        }
        this.infoLine();
        assert (binary.length() <= reserved);
    }

    private static void addEcc(int[] fullstream, int[] datastream, int version, int data_cw, int blocks) {
        int j;
        int ecc_cw = QR_TOTAL_CODEWORDS[version - 1] - data_cw;
        int short_data_block_length = data_cw / blocks;
        int qty_long_blocks = data_cw % blocks;
        int qty_short_blocks = blocks - qty_long_blocks;
        int ecc_block_length = ecc_cw / blocks;
        int[] data_block = new int[short_data_block_length + 2];
        int[] ecc_block = new int[ecc_block_length + 2];
        int[] interleaved_data = new int[data_cw + 2];
        int[] interleaved_ecc = new int[ecc_cw + 2];
        int posn = 0;
        for (int i = 0; i < blocks; ++i) {
            int length_this_block = i < qty_short_blocks ? short_data_block_length : short_data_block_length + 1;
            for (j = 0; j < ecc_block_length; ++j) {
                ecc_block[j] = 0;
            }
            for (j = 0; j < length_this_block; ++j) {
                data_block[j] = datastream[posn + j];
            }
            ReedSolomon rs = new ReedSolomon();
            rs.init_gf(285);
            rs.init_code(ecc_block_length, 0);
            rs.encode(length_this_block, data_block);
            for (j = 0; j < ecc_block_length; ++j) {
                ecc_block[j] = rs.getResult(j);
            }
            for (j = 0; j < short_data_block_length; ++j) {
                interleaved_data[j * blocks + i] = data_block[j];
            }
            if (i >= qty_short_blocks) {
                interleaved_data[short_data_block_length * blocks + (i - qty_short_blocks)] = data_block[short_data_block_length];
            }
            for (j = 0; j < ecc_block_length; ++j) {
                interleaved_ecc[j * blocks + i] = ecc_block[ecc_block_length - j - 1];
            }
            posn += length_this_block;
        }
        for (j = 0; j < data_cw; ++j) {
            fullstream[j] = interleaved_data[j];
        }
        for (j = 0; j < ecc_cw; ++j) {
            fullstream[j + data_cw] = interleaved_ecc[j];
        }
    }

    private static void setupGrid(int[] grid, int size, int version) {
        int i;
        boolean toggle = true;
        for (i = 0; i < size; ++i) {
            if (toggle) {
                grid[6 * size + i] = 33;
                grid[i * size + 6] = 33;
                toggle = false;
                continue;
            }
            grid[6 * size + i] = 32;
            grid[i * size + 6] = 32;
            toggle = true;
        }
        QrCode.placeFinder(grid, size, 0, 0);
        QrCode.placeFinder(grid, size, 0, size - 7);
        QrCode.placeFinder(grid, size, size - 7, 0);
        for (i = 0; i < 7; ++i) {
            grid[7 * size + i] = 16;
            grid[i * size + 7] = 16;
            grid[7 * size + (size - 1 - i)] = 16;
            grid[i * size + (size - 8)] = 16;
            grid[(size - 8) * size + i] = 16;
            grid[(size - 1 - i) * size + 7] = 16;
        }
        grid[7 * size + 7] = 16;
        grid[7 * size + (size - 8)] = 16;
        grid[(size - 8) * size + 7] = 16;
        if (version != 1) {
            int loopsize = QR_ALIGN_LOOPSIZE[version - 1];
            for (int x = 0; x < loopsize; ++x) {
                for (int y = 0; y < loopsize; ++y) {
                    int ycoord = QR_TABLE_E1[(version - 2) * 7 + y];
                    int xcoord = QR_TABLE_E1[(version - 2) * 7 + x];
                    if ((grid[ycoord * size + xcoord] & 0x10) != 0) continue;
                    QrCode.placeAlign(grid, size, xcoord, ycoord);
                }
            }
        }
        for (i = 0; i < 8; ++i) {
            int n = 8 * size + i;
            grid[n] = grid[n] + 32;
            int n2 = i * size + 8;
            grid[n2] = grid[n2] + 32;
            grid[8 * size + (size - 1 - i)] = 32;
            grid[(size - 1 - i) * size + 8] = 32;
        }
        int n = 8 * size + 8;
        grid[n] = grid[n] + 32;
        grid[(size - 1 - 7) * size + 8] = 33;
        if (version >= 7) {
            for (i = 0; i < 6; ++i) {
                grid[(size - 9) * size + i] = 32;
                grid[(size - 10) * size + i] = 32;
                grid[(size - 11) * size + i] = 32;
                grid[i * size + (size - 9)] = 32;
                grid[i * size + (size - 10)] = 32;
                grid[i * size + (size - 11)] = 32;
            }
        }
    }

    private static void placeFinder(int[] grid, int size, int x, int y) {
        int[] finder = new int[]{1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1};
        for (int xp = 0; xp < 7; ++xp) {
            for (int yp = 0; yp < 7; ++yp) {
                grid[(yp + y) * size + (xp + x)] = finder[xp + 7 * yp] == 1 ? 17 : 16;
            }
        }
    }

    private static void placeAlign(int[] grid, int size, int x, int y) {
        int[] alignment = new int[]{1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1};
        x -= 2;
        y -= 2;
        for (int xp = 0; xp < 5; ++xp) {
            for (int yp = 0; yp < 5; ++yp) {
                grid[(yp + y) * size + (xp + x)] = alignment[xp + 5 * yp] == 1 ? 17 : 16;
            }
        }
    }

    private static void populateGrid(int[] grid, int size, int[] fullstream, int cw) {
        boolean goingUp = true;
        int row = 0;
        int n = cw * 8;
        int y = size - 1;
        int i = 0;
        do {
            int x;
            if ((x = size - 2 - row * 2) < 6) {
                --x;
            }
            if ((grid[y * size + (x + 1)] & 0xF0) == 0) {
                grid[y * size + (x + 1)] = QrCode.cwbit(fullstream, i) ? 1 : 0;
                ++i;
            }
            if (i < n && (grid[y * size + x] & 0xF0) == 0) {
                grid[y * size + x] = QrCode.cwbit(fullstream, i) ? 1 : 0;
                ++i;
            }
            y = goingUp ? --y : ++y;
            if (y == -1) {
                ++row;
                y = 0;
                goingUp = false;
            }
            if (y != size) continue;
            ++row;
            y = size - 1;
            goingUp = true;
        } while (i < n);
    }

    private static boolean cwbit(int[] fullstream, int i) {
        return (fullstream[i / 8] & 128 >> i % 8) != 0;
    }

    private static int applyBitmask(int[] grid, int size, EccLevel ecc_level, StringBuilder encodeInfo) {
        int y;
        int x;
        byte[] mask = new byte[size * size];
        byte[] eval = new byte[size * size];
        for (x = 0; x < size; ++x) {
            for (y = 0; y < size; ++y) {
                mask[y * size + x] = 0;
                if ((grid[y * size + x] & 0xF0) != 0) continue;
                if ((y + x & 1) == 0) {
                    int n = y * size + x;
                    mask[n] = (byte)(mask[n] + 1);
                }
                if ((y & 1) == 0) {
                    int n = y * size + x;
                    mask[n] = (byte)(mask[n] + 2);
                }
                if (x % 3 == 0) {
                    int n = y * size + x;
                    mask[n] = (byte)(mask[n] + 4);
                }
                if ((y + x) % 3 == 0) {
                    int n = y * size + x;
                    mask[n] = (byte)(mask[n] + 8);
                }
                if ((y / 2 + x / 3 & 1) == 0) {
                    int n = y * size + x;
                    mask[n] = (byte)(mask[n] + 16);
                }
                if ((y * x & 1) + y * x % 3 == 0) {
                    int n = y * size + x;
                    mask[n] = (byte)(mask[n] + 32);
                }
                if (((y * x & 1) + y * x % 3 & 1) == 0) {
                    int n = y * size + x;
                    mask[n] = (byte)(mask[n] + 64);
                }
                if (((y + x & 1) + y * x % 3 & 1) != 0) continue;
                int n = y * size + x;
                mask[n] = (byte)(mask[n] + -128);
            }
        }
        for (x = 0; x < size; ++x) {
            for (y = 0; y < size; ++y) {
                int p = (grid[y * size + x] & 1) != 0 ? 255 : 0;
                eval[y * size + x] = (byte)(mask[y * size + x] ^ p);
            }
        }
        int best_pattern = 0;
        int best_val = Integer.MAX_VALUE;
        for (int pattern = 0; pattern < 8; ++pattern) {
            QrCode.addFormatInfoEval(eval, size, ecc_level, pattern);
            int penalty = QrCode.evaluate(eval, size, pattern, best_val, encodeInfo);
            if (penalty >= best_val) continue;
            best_pattern = pattern;
            best_val = penalty;
        }
        for (x = 0; x < size; ++x) {
            for (y = 0; y < size; ++y) {
                if ((mask[y * size + x] & 1 << best_pattern) == 0) continue;
                grid[y * size + x] = (grid[y * size + x] & 1) != 0 ? 0 : 1;
            }
        }
        return best_pattern;
    }

    private static void addFormatInfoEval(byte[] eval, int size, EccLevel ecc_level, int pattern) {
        int i;
        int format;
        switch (ecc_level.ordinal()) {
            case 0: {
                format = pattern | 8;
                break;
            }
            case 2: {
                format = pattern | 0x18;
                break;
            }
            case 3: {
                format = pattern | 0x10;
                break;
            }
            case 1: {
                format = pattern;
                break;
            }
            default: {
                throw new OkapiInternalException("Unknown ECC level: " + (Object)((Object)ecc_level));
            }
        }
        int seq = QR_ANNEX_C[format];
        for (i = 0; i < 6; ++i) {
            eval[i * size + 8] = (byte)(seq >> i & 1);
        }
        for (i = 0; i < 8; ++i) {
            eval[8 * size + (size - i - 1)] = (byte)(seq >> i & 1);
        }
        for (i = 0; i < 6; ++i) {
            eval[8 * size + (5 - i)] = (byte)(seq >> i + 9 & 1);
        }
        for (i = 0; i < 7; ++i) {
            eval[(size - 7 + i) * size + 8] = (byte)(seq >> i + 8 & 1);
        }
        eval[7 * size + 8] = (byte)(seq >> 6 & 1);
        eval[8 * size + 8] = (byte)(seq >> 7 & 1);
        eval[8 * size + 7] = (byte)(seq >> 8 & 1);
    }

    private static int evaluate(byte[] eval, int size, int pattern, int best, StringBuilder encodeInfo) {
        boolean light;
        int y;
        int block;
        byte state;
        int x;
        int i;
        int result = 0;
        byte[] local = new byte[size * size];
        int dark_mods = 0;
        int mask = 1 << pattern;
        for (i = 0; i < local.length; ++i) {
            byte val;
            local[i] = val = (byte)((eval[i] & mask) >> pattern);
            dark_mods += val;
        }
        encodeInfo.append("Mask ").append(QrCode.maskToString(pattern)).append(" Penalties: ");
        for (x = 0; x < size; ++x) {
            state = local[x];
            block = 1;
            for (y = 1; y < size; ++y) {
                i = y * size + x;
                if (local[i] == state) {
                    ++block;
                    continue;
                }
                if (block > 5) {
                    result += 3 + (block - 5);
                }
                block = 0;
                state = local[i];
            }
            if (block <= 5) continue;
            result += 3 + (block - 5);
        }
        for (y = 0; y < size; ++y) {
            state = local[y * size];
            block = 1;
            for (x = 1; x < size; ++x) {
                i = y * size + x;
                if (local[i] == state) {
                    ++block;
                    continue;
                }
                if (block > 5) {
                    result += 3 + (block - 5);
                }
                block = 0;
                state = local[i];
            }
            if (block <= 5) continue;
            result += 3 + (block - 5);
        }
        encodeInfo.append(result).append(' ');
        if (result > best) {
            encodeInfo.append("EXIT\n");
            return result;
        }
        for (x = 0; x < size - 1; ++x) {
            for (y = 0; y < size - 1; ++y) {
                i = y * size + x;
                state = local[i];
                if (state != local[i + 1] || state != local[i + size] || state != local[i + size + 1]) continue;
                result += 3;
            }
        }
        encodeInfo.append(result).append(' ');
        if (result > best) {
            encodeInfo.append("EXIT\n");
            return result;
        }
        for (x = 0; x < size; ++x) {
            for (y = 0; y < size - 7; ++y) {
                if (local[(y + 0) * size + x] != 1 || local[(y + 1) * size + x] != 0 || local[(y + 2) * size + x] != 1 || local[(y + 3) * size + x] != 1 || local[(y + 4) * size + x] != 1 || local[(y + 5) * size + x] != 0 || local[(y + 6) * size + x] != 1) continue;
                light = true;
                for (i = y - 4; i < y; ++i) {
                    if (i < 0 || local[i * size + x] != 1) continue;
                    light = false;
                    break;
                }
                if (light) {
                    result += 40;
                    continue;
                }
                light = true;
                for (i = y + 7; i <= y + 10; ++i) {
                    if (i >= size || local[i * size + x] != 1) continue;
                    light = false;
                    break;
                }
                if (!light) continue;
                result += 40;
            }
        }
        for (y = 0; y < size; ++y) {
            for (x = 0; x < size - 7; ++x) {
                if (local[y * size + x + 0] != 1 || local[y * size + x + 1] != 0 || local[y * size + x + 2] != 1 || local[y * size + x + 3] != 1 || local[y * size + x + 4] != 1 || local[y * size + x + 5] != 0 || local[y * size + x + 6] != 1) continue;
                light = true;
                for (i = x - 4; i < x; ++i) {
                    if (i < 0 || local[y * size + i] != 1) continue;
                    light = false;
                    break;
                }
                if (light) {
                    result += 40;
                    continue;
                }
                light = true;
                for (i = x + 7; i <= x + 10; ++i) {
                    if (i >= size || local[y * size + i] != 1) continue;
                    light = false;
                    break;
                }
                if (!light) continue;
                result += 40;
            }
        }
        encodeInfo.append(result).append(' ');
        if (result > best) {
            encodeInfo.append("EXIT\n");
            return result;
        }
        int percentage = (int)(100.0 * (double)dark_mods / (double)(size * size));
        int k = Math.abs(percentage - 50) / 5;
        encodeInfo.append(result += 10 * k).append('\n');
        return result;
    }

    private static void addFormatInfo(int[] grid, int size, EccLevel ecc_level, int pattern) {
        int i;
        int format;
        switch (ecc_level.ordinal()) {
            case 0: {
                format = pattern | 8;
                break;
            }
            case 2: {
                format = pattern | 0x18;
                break;
            }
            case 3: {
                format = pattern | 0x10;
                break;
            }
            case 1: {
                format = pattern;
                break;
            }
            default: {
                throw new OkapiInternalException("Unknown ECC level: " + (Object)((Object)ecc_level));
            }
        }
        int seq = QR_ANNEX_C[format];
        for (i = 0; i < 6; ++i) {
            int n = i * size + 8;
            grid[n] = grid[n] + (seq >> i & 1);
        }
        for (i = 0; i < 8; ++i) {
            int n = 8 * size + (size - i - 1);
            grid[n] = grid[n] + (seq >> i & 1);
        }
        for (i = 0; i < 6; ++i) {
            int n = 8 * size + (5 - i);
            grid[n] = grid[n] + (seq >> i + 9 & 1);
        }
        for (i = 0; i < 7; ++i) {
            int n = (size - 7 + i) * size + 8;
            grid[n] = grid[n] + (seq >> i + 8 & 1);
        }
        int n = 7 * size + 8;
        grid[n] = grid[n] + (seq >> 6 & 1);
        int n2 = 8 * size + 8;
        grid[n2] = grid[n2] + (seq >> 7 & 1);
        int n3 = 8 * size + 7;
        grid[n3] = grid[n3] + (seq >> 8 & 1);
    }

    private static void addVersionInfo(int[] grid, int size, int version) {
        int version_data = QR_ANNEX_D[version - 7];
        for (int i = 0; i < 6; ++i) {
            int n = (size - 11) * size + i;
            grid[n] = grid[n] + (version_data >> i * 3 & 1);
            int n2 = (size - 10) * size + i;
            grid[n2] = grid[n2] + (version_data >> i * 3 + 1 & 1);
            int n3 = (size - 9) * size + i;
            grid[n3] = grid[n3] + (version_data >> i * 3 + 2 & 1);
            int n4 = i * size + (size - 11);
            grid[n4] = grid[n4] + (version_data >> i * 3 & 1);
            int n5 = i * size + (size - 10);
            grid[n5] = grid[n5] + (version_data >> i * 3 + 1 & 1);
            int n6 = i * size + (size - 9);
            grid[n6] = grid[n6] + (version_data >> i * 3 + 2 & 1);
        }
    }

    protected void customize(int[] grid, int size) {
    }

    public static enum EccLevel {
        L,
        M,
        Q,
        H;

    }

    private static enum QrMode {
        NULL,
        KANJI,
        BINARY,
        ALPHANUM,
        NUMERIC;

    }
}

